
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   MSG1100.C - Message 1100 Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode, build, and display Zodiac 1100 messages.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\MSG1110.C_V  $
//   
//      Rev 1.15   09 Nov 1998 10:47:50   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.3   Jul 09 1997 09:57:40   COLEJ
//   Multiple Changes...
//   
//      Rev 1.2   Feb 12 1997 16:05:24   COLEJ
//    
//   
//      Rev 1.1   Aug 14 1996 10:02:36   COLEJ
//    
//   
//      Rev 1.0   13 May 1996 14:52:52   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <share.h>
#include <math.h>

#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"   
#include "util.h" 
#include "display.h"

void wait(float);  
void clear_message_line(void);

extern FILE  *stream5;
extern short  StoreFrqPar;
extern char   frqparfile[32];

tFLOAT  TRef;
tDOUBLE S0;
tUSHORT T0;
tUSHORT MSG1110; // message on flag
////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1110 message from the ZMsgBuf.
//
void Dec1110(tMSGBUF *ZMsgBuf, tMSG1110 *Msg)
{
   tDOUBLE C0, C1, C2, C3;
   tFLOAT  TInf;
   tDOUBLE U0, U1;

   GetShort  (&Msg->SequenceNumber      , ZMsgBuf, 8);  // ND   
   GetUShort (&Msg->FSIssue             , ZMsgBuf, 9);  // ND 
   GetShort  (&Msg->Characteristics.C0  , ZMsgBuf,10);  // 2^29 * sec/sec 
   GetShort  (&Msg->Characteristics.C1  , ZMsgBuf,11);  // 2^35 * sec/sec/deg C 
   GetShort  (&Msg->Characteristics.C2  , ZMsgBuf,12);  // 2^41 * sec/sec/deg C^2 
   GetShort  (&Msg->Characteristics.C3  , ZMsgBuf,13);  // 2^47 * sec/sec/deg C^3 
   GetShort  (&Msg->Characteristics.TInf, ZMsgBuf,14);  // 100 * deg C
   GetShort  (&Msg->Dynamics.D0         , ZMsgBuf,15);  // ND 
   GetShort  (&Msg->Dynamics.D1         , ZMsgBuf,16);  // ND 
   GetShort  (&Msg->Calibration.TRef    , ZMsgBuf,17);  // 100 * deg C 
   GetUShort (&Msg->Calibration.T0      , ZMsgBuf,18);  // counts
   GetShort  (&Msg->Calibration.S0      , ZMsgBuf,19);  // 2^18 * deg C/count     
   GetShort  (&Msg->Uncertainty.U0      , ZMsgBuf,20);  // 2^29 * sec/sec     
   GetShort  (&Msg->Uncertainty.U1      , ZMsgBuf,21);  // 2^35 * sec/sec/deg C 
   
   // store frequency parameters in a file
   if(StoreFrqPar && (stream5 = _fsopen(frqparfile, "wt", SH_DENYWR)) != NULL){
   clear_message_line();
   ShowText("STORING FREQUENCY STANDARD PARAMETERS IN FILE ... ", CMDR, CMDC);
   StoreFrqPar = 0;

   C0   = (tDOUBLE)(Msg->Characteristics.C0  / TWOP29);
   C1   = (tDOUBLE)(Msg->Characteristics.C1  / TWOP35);
   C2   = (tDOUBLE)(Msg->Characteristics.C2  / TWOP41);
   C3   = (tDOUBLE)(Msg->Characteristics.C3  / TWOP47);

   TInf = (tFLOAT)(Msg->Characteristics.TInf / 100.0);
   TRef = (tFLOAT)(Msg->Calibration.TRef     / 100.0);

   S0   = (tDOUBLE)(Msg->Calibration.S0      / TWOP18);
   U0   = (tDOUBLE)(Msg->Uncertainty.U0      / TWOP29);
   U1   = (tDOUBLE)(Msg->Uncertainty.U1      / TWOP35);

   fprintf(stream5,"FREQUENCY STANDARD PARAMETERS\n");
   fprintf(stream5,"FILENAME       : %s\n\n"  , frqparfile);
   fprintf(stream5,"SEQ NUMBER     : %-8d\n"  , Msg->SequenceNumber);
   fprintf(stream5,"FREQ STD ISSUE : %-5u\n"  , Msg->FSIssue);
   fprintf(stream5,"C0             : %-.15lg\n", C0);
   fprintf(stream5,"C1             : %-.15lg\n", C1);
   fprintf(stream5,"C2             : %-.15lg\n", C2);
   fprintf(stream5,"C3             : %-.15lg\n", C3);
   fprintf(stream5,"TINF           : %-8.2f\n", TInf);
   fprintf(stream5,"D0             : %-.15lg\n", Msg->Dynamics.D0);
   fprintf(stream5,"D1             : %-.15lg\n", Msg->Dynamics.D1);
   fprintf(stream5,"TREF           : %-8.2f\n", TRef);
   fprintf(stream5,"T0             : %-8u\n"  , Msg->Calibration.T0);
   fprintf(stream5,"S0             : %-.15lg\n", S0);
   fprintf(stream5,"U0             : %-.15lg\n", U0);
   fprintf(stream5,"U1             : %-.15lg\n", U1);

   fclose(stream5);
   ShowText("DONE", CMDR, CMDC+50);
   }
   // get data for temp calculation in msg1102
   MSG1110 = 1; // turn flag on
   TRef = (tFLOAT)(Msg->Calibration.TRef     / 100.0);
   T0   = (Msg->Calibration.T0);
   S0   = (tDOUBLE)(Msg->Calibration.S0      / TWOP18);

}                                      
                                     
////////////////////////////////////////////////////////////////////////////////
//   
//   Build a Zodiac 1110 message in the ZMsgBuf. 
//
void Bld1110(tMSGBUF *ZMsgBuf, tMSG1110 *Msg)
{
   // ack flags assumed preset

   PutShort  (0x81FF                    , ZMsgBuf, 1);     // message preamble
   PutShort  (1110                      , ZMsgBuf, 2);     // message number
   PutShort  (16                        , ZMsgBuf, 3);     // message length 
   PutShort  (ANREQ                     , ZMsgBuf, 4);     // message ack/nack request
 //PutShort  (CheckSum(ZMsgBuf[0], 4)   , ZMsgBuf, 5);     // header check sum
                                                  
   PutShort  (Msg->SequenceNumber       , ZMsgBuf, 8);  // ND   
   PutUShort (Msg->FSIssue              , ZMsgBuf, 9);  // ND 
   PutShort  (Msg->Characteristics.C0   , ZMsgBuf,10);  // 2^29 * sec/sec 
   PutShort  (Msg->Characteristics.C1   , ZMsgBuf,11);  // 2^35 * sec/sec/deg C 
   PutShort  (Msg->Characteristics.C2   , ZMsgBuf,12);  // 2^41 * sec/sec/deg C^2 
   PutShort  (Msg->Characteristics.C3   , ZMsgBuf,13);  // 2^47 * sec/sec/deg C^3 
   PutShort  (Msg->Characteristics.TInf , ZMsgBuf,14);  // 100 * deg C
   PutShort  (Msg->Dynamics.D0          , ZMsgBuf,15);  // ND 
   PutShort  (Msg->Dynamics.D1          , ZMsgBuf,16);  // ND 
   PutShort  (Msg->Calibration.TRef     , ZMsgBuf,17);  // 100 * deg C 
   PutUShort (Msg->Calibration.T0       , ZMsgBuf,18);  // counts
   PutShort  (Msg->Calibration.S0       , ZMsgBuf,19);  // 2^18 * deg C/count     
   PutShort  (Msg->Uncertainty.U0       , ZMsgBuf,20);  // 2^29 * sec/sec     
   PutShort  (Msg->Uncertainty.U1       , ZMsgBuf,21);  // 2^35 * sec/sec/deg C     

 //PutShort  (CheckSum(ZMsgBuf, 6, 16)       , ZMsgBuf, 22);    // data check sum 
}

////////////////////////////////////////////////////////////////////////////////
//   
//   Display a Zodiac 1110 message from the ZMsgBuf.
//
void Show1110(tMSG1110 *Msg)
{      
   // tbd
}                                           
